/** @file   overlayitem.h
 * @brief   Declaration of OverlayItem - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */
 

#ifndef H_WWW_OVERLAYITEM_H
#define H_WWW_OVERLAYITEM_H

#include "vec2d.h"

// FOrward declarations
struct BITMAP;

namespace WeWantWar {

/** @class  OverlayItem
 * @brief   Baseclass for transparent popup dialogs you can overlay on the
 *          screen.
 * @author  Tomi Lamminsaari
 *
 * This class provides the interface and background buffer features.
 */
class OverlayItem
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================
  
  /** Datatype for opaque - values. */
  typedef int OpacityVal;
  

  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   * @param     x                 X coordinate of the topleft corner.
   * @param     y                 Y coordinate of the topleft corner
   * @param     w                 Width of this item
   * @param     h                 Height of this item.
   */
	OverlayItem( int x, int y, int w, int h );


	/** Destructor
   */
	virtual ~OverlayItem();
	
private:
	/** Copy constructor. DISABLED !
   * @param     rO                Reference to another OverlayItem
   */
  OverlayItem( const OverlayItem& rO );

	/** Assignment operator. DISABLED !
   * @param     rO                Reference to another OverlayItem
   * @return    Reference to us.
   */
  OverlayItem& operator = ( const OverlayItem& rO );

public:

  ///
  /// Methods
  /// =======

  /** Sets the opaque-value for this OverlayItem.
   * @param     opa               Opaque. 0 = transparent, 255 = opaque
   */
  void opacity( OpacityVal opa );
  
  /** Positions this OverlayItem
   * @param     rPos              New position
   */
  void position( const eng2d::Vec2D& rPos );

  /** Updates the contents, animations, etc. of this item. Declared pure
   * virtual so this needs to be implemented in child classes.
   *
   * The implementation of this method should prepare the background buffer
   * because the default implementation of the <code>redraw( BITMAP* pB )</code>
   * just draws it on the target bitmap.
   *
   * Use <code> getBuffer() </code> - method to get pointer to the background
   * buffer.
   */
  virtual void update() = 0;

  /** Draws the contents of this item on the given bitmap. This respects
   * the opaque-settings while drawing.
   * @param     pB                Pointer to target bitmap.
   */
  virtual void redraw( BITMAP* pB ) const;


  /** Fades this OverlayItem towards the visibility. The opacity-value
   * is increased by given factor.
   * @param     opa               Amount of opacity change
   */
  void fadeIn( OpacityVal opa );
  
  /** Fades this OverlayItem towards the complete transparency.
   * @param     opa               How much the opacity will be reduced
   */
  void fadeOut( OpacityVal opa );

  /** Sets the minimum and maximum opacity values.
   * @param     minOpa            Minimum opacity
   * @param     maxOpa            Maximum opacity
   */
  void setOpacityRange( OpacityVal minOpa, OpacityVal maxOpa );


  ///
  /// Getter methods
  /// ==============
  
  /** Returns the opaque-value.
   * @return    Opaque ranging from 0 to 255. 0 = transparent
   */
  OpacityVal opacity() const;
  
  /** Returns the position of this OverlayItem.
   * @return    The position
   */
  eng2d::Vec2D position() const;


protected:

  ///
  /// Protected interface
  /// ===================
  
  /** Returns the pointer to background buffer.
   * @return    Pointer to background buffer.
   */
  BITMAP* getBuffer() const;

  ///
  /// Members
  /// =======

  /** The transparency of this OverlayItem. 0 = transparent, 256 = opaque */
  int   m_opacity;
  
  /** The bitmap we use as a background buffer. */
  BITMAP* m_pBuffer;
  
  /** Position where the topleft corner of this item is located. */
  eng2d::Vec2D  m_position;
  
  /** The maximum opacity-value */
  OpacityVal  m_maxOpacity;
  
  /** The minimum opacity value */
  OpacityVal  m_minOpacity;

private:

  ///
  /// Private members
  /// ===============

};


};  // end of namespace
#endif

/**
 * Version history
 * ===============
 * $Log: overlayitem.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.1  2005-11-13 14:26:04+02  lamminsa
 * redraw made virtual.
 *
 */
 
